/* Copyright (C) 2002-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef __VNCMIRRORLINK_H__
#define __VNCMIRRORLINK_H__

/**
 * \file vncmirrorlink.h
 *
 * This file contains function definitions, additional key symbols and other
 * constants that are specifically for use with MirrorLink&reg; VNC Servers.
 *
 * <B>IMPORTANT.</B>  In order to enable MirrorLink support within the Viewer
 * SDK, you must first set VNCParameterMirrorLink to a non-zero value. Failure
 * to do so will cause both the Viewer SDK and the MirrorLink server to treat
 * the session as a regular RFB 3 session.  In that case, even if the Viewer
 * SDK is licensed to connect to regular MirrorLink servers, MirrorLink
 * functionality would not be available.
 *
 * MirrorLink is a registered trademark of Car Connectivity Consortium in the
 * U.S. and in other countries.
 *
 * \see VNCViewerSetParameter, VNCParameterMirrorLink
 *
 * \section mirrorlink_contents Contents
 *
 *  -# \ref mirrorlink_keys MirrorLink device keys
 *  -# \ref mirrorlink_extensions MirrorLink extension messages
 *  -# \ref mirrorlink_context Context Information and driver distraction
 *  -# \ref mirrorlink_rle Scan-line based Run-Length Encoding
 *  -# \ref mirrorlink_vah264 VA H.264 Encoding
 *  -# \ref mirrorlink_virtual_kbd Virtual Keyboard Triggers
 *  -# \ref mirrorlink_event_mapping Event Mapping
 *  -# \ref mirrorlink_key_event_listing Key Event Listing
 *  -# \ref mirrorlink_ca Content attestation
 *  -# \ref mirrorlink_references References
 *  -# \ref mirrorlink_legal Legal information
 *
 * \section mirrorlink_keys MirrorLink device keys
 *
 * MirrorLink defines additional key symbols for use with MirrorLink servers.
 * These key symbols can be found in mirrorlinkkeysymdef.h.
 *
 * The SDK provides several APIs for sending keyboard input events to VNC
 * Servers.  MirrorLink key symbols can be sent by calling
 * VNCViewerSendKeyEvent(), which expects a key symbol (MirrorLink or regular)
 * as its parameter, or by calling VNCViewerSendDeviceKeyEvent(), which expects
 * a member of the VNCDeviceKey enumeration as its parameter.
 *
 * VNCViewerSendDeviceKeyEvent() automatically selects the correct key to send
 * to the currently connected server.  Therefore it is recommended to use this
 * API where the same viewer application is required to interoperate with both
 * VNC Mobile Solution servers and MirrorLink servers.  If the viewer
 * application is required to interoperate only with MirrorLink servers, then
 * VNCViewerSendKeyEvent() is sufficient.
 *
 * VNCViewerSendDeviceKeyOrKeyEvent() is similar to VNCViewerSendKeyEvent(),
 * except that it maps some keyboard keys to important device keys, such as the
 * 'Home' key to the home device key.  Under Windows and Windows CE,
 * VNCViewerProcessWindowsKeyMessage() is a convenient alternative to
 * VNCViewerSendKeyEvent() or VNCViewerSendDeviceKeyOrKeyEvent().
 *
 * \see mirrorlinkkeysymdef.h, VNCViewerSendKeyEvent,
 * VNCViewerSendDeviceKeyEvent
 *
 * \section mirrorlink_extensions MirrorLink extension messages
 *
 * MirrorLink adds additional messages, known as MirrorLink extensions, to the
 * RFB protocol.  These messages allow for tighter integration between the
 * viewer and server, with a focus on automotive use cases.
 *
 * The SDK provides sensible default handling for all of the mandatory
 * MirrorLink extension messages that the server may send.  However, it is
 * likely that the viewer application will need to be informed about some of
 * these extension messages.  This also allows the viewer application to
 * override the default behaviour of some of these message.
 *
 * \subsection mirrorlink_handshake The MirrorLink handshake
 *
 * Shortly after the main RFB handshake is complete, a MirrorLink session
 * always exchanges the following extension messages:
 *
 * \verbatim
     Viewer                            Server
     ------                            ------
                                    <- ServerDisplayConfiguration
     ClientDisplayConfiguration ->   
                                    <- ServerEventConfiguration
     ClientEventConfiguration   ->   
   \endverbatim
 * 
 * The ServerDisplayConfiguration message describes the server's capabilities
 * with respect to its display - whether it is capable of scaling it or
 * rotating it, which pixel formats it is capable of serving, and so on.  The
 * viewer's response to this is the ClientDisplayConfiguration message, which
 * informs the server which of these capabilities the viewer will use.
 *
 * The ServerEventConfiguration message describes the server's supported input
 * mechanisms - whether it understands knob input, ITU keypad input, multimedia
 * keys, touchscreen input, and so on.  Again, the viewer's response to this,
 * the ClientEventConfiguration message, informs the server which of these
 * capabilities the viewer will use.
 *
 * The SDK decodes the ServerDisplayConfiguration message into a
 * VNCServerDisplayConfiguration structure and passes it to the viewer
 * application via the optional VNCViewerServerDisplayConfigurationCallback.
 * This callback also provides the viewer application with a
 * VNCClientDisplayConfiguration structure that has been pre-filled with
 * sensible default values.  the viewer application may alter any of the fields
 * in the structure before the callback returns, at which point the SDK will
 * encode it and sent it to the server.
 *
 * Similarly, the optional VNCViewerServerEventConfigurationCallback provides
 * the application with the decoded ServerEventConfiguration message in a
 * VNCServerEventConfiguration structure and also a pre-filled
 * VNCClientEventConfiguration structure, whose fields the viewer application
 * is free to alter.  Additionally, this callback provides the viewer
 * application with the first VNCDeviceStatusRequest (see
 * \ref mirrorlink_status "DeviceStatus and DeviceStatusRequest messages").
 *
 * \see VNCViewerServerDisplayConfigurationCallback,
 * VNCViewerServerEventConfigurationCallback
 *
 * \subsection mirrorlink_status DeviceStatus and DeviceStatusRequest messages
 *
 * MirrorLink adds the ability for the viewer to instruct the server to
 * reconfigure certain elements of its functionality from within the session.
 * For example, the viewer may instruct the server to switch between portrait
 * and landscape orientations, or to enable or disable the key lock on the
 * device.  These instructions are sent to the server in a DeviceStatusRequest
 * extension message.
 *
 * At any time after the handshake is complete, the viewer may send a
 * DeviceStatusRequest message to the server.  The server should respond with a
 * DeviceStatus message:
 *
 * \verbatim
     Viewer                            Server
     ------                            ------
     DeviceStatusRequest        ->
                                    <- DeviceStatus
   \endverbatim
 *
 * The Viewer SDK always sends at least one DeviceStatusRequest to the server
 * in every MirrorLink session.  This is so that the server can be put into a
 * known orientation, which is necessary in order for the SDK to know whether
 * direction button input should be rotated to match that orientation.
 *
 * This initial DeviceStatusRequest is sent immediately after the
 * ServerEventConfiguration message in the handshake (see
 * \ref mirrorlink_handshake "The MirrorLink handshake").
 * It is supplied to the VNCViewerServerEventConfigurationCallback as a
 * pre-filled VNCDeviceStatusRequest structure, whose fields the application is
 * free to override.  Thereafter, a DeviceStatusRequest may be sent at any time
 * by calling VNCViewerSendDeviceStatusRequest.
 *
 * Some device features have been deprecated in MirrorLink 1.3. The viewer
 * application should not attempt to use deprecated feature status values if it
 * supports MirrorLink 1.3 or above. See \ref VNCDeviceStatusFeature for
 * further information.
 *
 * Each received DeviceStatus message, including the response to the initial
 * DeviceStatusRequest, results in a call to the optional
 * \ref VNCViewerDeviceStatusCallback.  This informs the viewer application
 * whether the server has been able to honour the request.
 *
 * \see VNCViewerServerEventConfigurationCallback,
 * VNCViewerSendDeviceStatusRequest,
 * VNCViewerDeviceStatusCallback
 *
 * \subsection mirrorlink_blocking FramebufferBlockingNotification and AudioBlockingNotification messages
 *
 * \verbatim
     Viewer                              Server
     ------                              ------
     FramebufferBlockingNotification ->

     AudioBlockingNotification       ->
   \endverbatim
 *
 * FramebufferBlockingNotifications are unsolicited messages sent by the viewer
 * to the server.  They inform the server that the viewer application is unable
 * or unwilling to render part of the framebuffer.  The reason for this is
 * given by the blockReason member of the VNCFramebufferBlockingNotification
 * structure that the viewer application supplies to the
 * VNCViewerSendFramebufferBlockingNotification() API.
 *
 * Sending a FramebufferBlockingNotification may cause the server to hide the
 * application that has been blocked.  A good use for this feature is to call
 * VNCViewerSendFramebufferBlockingNotification() from within the
 * viewer application's VNCViewerContextInformationCallback when the
 * application being run on the server does not comply with the driver
 * distraction policy (see
 * \ref mirrorlink_context "Context Information and driver distraction").
 *
 * Similarly, AudioBlockingNotification messages inform the server that the
 * viewer application is unable or unwilling to render audio that the server is
 * streaming.  The reason for this is given by the blockReason member of the
 * VNCAudioBlockingNotification structure that the viewer application supplies
 * to the VNCViewerSendAudioBlockingNotification() API.  Again, the reason may
 * be that the audio does not comply with a driver distraction policy, or it
 * may simply be that the user has muted the audio output device.
 *
 * AudioBlockingNotification messages are also used to inform the server that
 * an audio stream should be resumed, by setting the blockReason member of the
 * VNCAudioBlockingNotification structure to VNCAudioBlockReasonUnblock.
 *
 * \see VNCViewerSendFramebufferBlockingNotification,
 * VNCViewerSendAudioBlockingNotification
 *
 * \section mirrorlink_context Context Information and driver distraction
 *
 * MirrorLink servers provide the viewer with information about the
 * applications that are visible on the server display.  This is called Context
 * Information and is intended to allow the viewer application selectively to
 * filter or block these applications - for example, for compliance with the
 * viewer application's driver distraction policy.
 *
 * The SDK provides Context Information to the viewer application via the
 * optional VNCViewerContextInformationCallback.  If the viewer application
 * decides that it should refuse to render or even dismiss a server
 * application, then it may wish to send a FramebufferBlockingNotification
 * message to the server (see
 * \ref mirrorlink_blocking "FramebufferBlockingNotification and AudioBlockingNotification messages").
 *
 * \see VNCViewerContextInformationCallback
 *
 * \section mirrorlink_rle Scan-line based Run-Length Encoding
 *
 * MirrorLink defines a new encoding for pixel data.  The Viewer SDK has full
 * support for this encoding, which is known as SLRLE in the API.
 *
 * Note that, Nokia's prototype MirrorLink server versions before 2.7 contain a
 * bug in the SLRLE implementation which will cause the SDK to be unable to
 * parse the encoded data.
 *
 * This means that only Nokia's prototype MirrorLink servers with a version of
 * 2.7 or greater are supported by this SDK.
 *
 * \see VNCViewerSetParameter(), VNCParameterPreferredEncoding, VNCEncoding
 *
 * \section mirrorlink_vah264 VA H.264 Encoding
 *
 * MirrorLink 1.3 adopts the VA H.264 RFB encoding. The Viewer SDK can support
 * this encoding through the pluggable decoder and the pluggable renderer.
 *
 * \see vncdecoder.h, vncrenderer.h
 *
 * \section mirrorlink_virtual_kbd Virtual Keyboard Triggers
 *
 * MirrorLink servers can provide the viewer with the ability to be notified
 * when a virtual keyboard should be shown. This allows viewers to
 * display a virtual keyboard in response to a text entry field being
 * selected on the server. Note that this is not supported in MirrorLink 1.3
 * sessions and above.
 *
 * To request these notification from the server a viewer application
 * needs to call VNCViewerSendVirtualKeyboardTriggerRequest() with an
 * appropriate configuration.
 *
 * The SDK provides virtual keyboard trigger notifications to the viewer
 * application via the optional VNCViewerVirtualKeyboardTriggerCallback.
 *
 * If the viewer application wishes to disable these notifications then it
 * should call VNCViewerSendVirtualKeyboardTriggerRequest() with an
 * appropriate configuration.
 *
 * If the viewer application has enabled support for text entry
 * exchange and has received confirmation of support from the server
 * then the viewer application is responsible for using
 * VNCViewerSendClientCutText() and responding to the
 * VNCViewerServerCutTextCallback appropriately.
 *
 * \see VNCViewerVirtualKeyboardTriggerCallback
 *
 * \section mirrorlink_event_mapping Event Mapping
 *
 * MirrorLink servers can provide event mapping of certain key symbol values.
 * Note that this is not supported in MirrorLink 1.3 sessions and above.
 *
 * To request an event remapping the viewer application needs to call
 * VNCViewerSendEventMappingRequest() with the appropriate details.
 *
 * The response from the server for this request will be received via the
 * optional VNCViewerEventMappingCallback.
 *
 * The server can also send a VNCViewerEventMappingCallback when the mapping
 * changes for other reasons.
 *
 * Before using EventMapping a viewer application should check the Client Event 
 * Configuration message for support.
 *
 * The SDK will perform no remapping of events itself, it is the
 * responsibility of the viewer application to handle remapping
 * requests and callbacks appropriately.
 *
 * \see VNCViewerEventMappingCallback
 *
 * \section mirrorlink_key_event_listing Key Event Listing
 *
 * MirrorLink servers can provide the viewer application with a list
 * of valid key events. Note that this is not supported in MirrorLink 1.3
 * sessions and above.
 *
 * To request this list of events a viewer application needs to call
 * VNCViewerSendKeyEventListingRequest() with the appropriate details.
 *
 * The SDK will process the Key Event Listing responses from the server,
 * maintaining a list of the valid key symbols. The SDK will inform the
 * viewer application of this list via the optional
 * VNCViewerKeyEventListingCallback.
 *
 * \see VNCViewerKeyEventListingCallback
 *
 * \section mirrorlink_ca Content attestation
 *
 * The SDK can automatically perform MirrorLink content attestation if
 * configured to do so. By default, content attestation is not performed.
 *
 * \subsection mirrorlink_ca_config Configuration
 *
 * To enable content attestation, the application must set BOTH of these two
 * parameters (using VNCViewerSetParameter()):
 *
 *  -# ::VNCParameterContentAttestationSignatureFlag. The value should be the
 *     value of the signature flag field of ContentAttestationRequest messages.
 *     It controls which factors the server attests. The value is a combination
 *     of the bits defined for the signature flag in section 5.8 (v1.1 and
 *     v1.2) or 5.6 (v1.3) of \ref ml_vnc "[ML-VNC]":
 *
 *      - bit 0 (value 1) - server should attest context information
 *      - bit 1 (value 2) - server should attest framebuffer content
 *      - bit 2 (value 4) - server should attest the number of updated pixels
 *    
 *  -# ::VNCParameterMirrorLinkVNCServerPublicKey. This is the public key of
 *     the TerminalMode:VNC-Server component, as returned by the Device
 *     Attestation Protocol. If you are using the VNC DAP SDK, then this the
 *     applicationPublicKeyBase64 string in the attestation for
 *     VNCDAPComponentTerminalModeVNCServer.
 *
 * To request that the server attests all three factors, set
 * VNCParameterContentAttestationSignatureFlag to "7". The default value is "0",
 * which disables content attestation.
 *
 * \subsection mirrorlink_ca_callback Handling content attestation failures
 *
 * The application may also provide a
 * ::VNCViewerContentAttestationFailureCallback. This callback is called when:
 *
 *  - The server does not support content attestation.
 *  - The server did not respond to a ContentAttestationRequest in a timely
 *    manner.
 *  - The server did not sign its ContentAttestationResponse.
 *  - The server did not attest one of the factors that the client requested it
 *    should. For example, if ::VNCParameterContentAttestationSignatureFlag is
 *    "3", then the client application has requested that the server should
 *    attest the framebuffer context information and the framebuffer content.
 *    However, if the server only attests the framebuffer content, and does not
 *    attest the context information, then the SDK will invoke the callback.
 *
 * Some MirrorLink client implementations may consider one or more of these
 * error conditions to be non-fatal. In that case, the application should
 * implement a VNCViewerContentAttestationFailureCallback. The callback should
 * inspect the failure parameter to the callback, and then call
 * VNCViewerContentAttestationFailureResult() to let the SDK know to proceed.
 * The call to VNCViewerContentAttestationFailureResult() may be made either
 * from within the callback, or at a later time (e.g. after consulting the end
 * user).
 *
 * The ::VNCViewerContentAttestationFailureCallback is optional. If no callback
 * is provided, then the SDK considers any content attestation failure to be
 * fatal.
 *
 * \subsection mirrorlink_ca_fatal Fatal content attestation failures
 *
 * The SDK does not invoke the callback for error conditions that are always
 * fatal. For example:
 *
 *  - A ContentAttestationResponse contained a signature, but the signature was
 *    incorrect.
 *  - The server attested a requested factor (e.g. the framebuffer content), but
 *    the hash of the attested data was incorrect.
 *
 * In these cases, the SDK always terminates the session without calling the
 * callback.
 *
 * \subsection mirrorlink_ca_error Session termination
 *
 * If a session is terminated due to a content attestation failure, either
 * because there was a fatal failure or because the application requested the
 * termination with VNCViewerContentAttestationFailureResult(), then the error
 * code reported by the ::VNCViewerErrorCallback is
 * ::VNCViewerErrorContentAttestationFailed.
 *
 * \subsection mirrorlink_hsml High Speed Media Link
 *
 * In MirrorLink 1.2 and above, a High Speed Media Link (HSML) connection may
 * be used to convey framebuffer pixel data. This operates alongside the VNC
 * session, which is used to provide a control channel for user input events,
 * reporting of context information and other tasks. HSML is described further
 * in \ref ml_hsml "[ML-HSML]".
 *
 * If the viewer application supports HSML, then the VNCParameterHSML parameter
 * should be set to true before establishing a VNC session. This will cause the
 * viewer SDK to advertise the HSML pseudo-encoding to the MirrorLink server in
 * RFB 3 MirrorLink connections. If the MirrorLink server also supports HSML,
 * then it shall include an HSML information rectangle in the framebuffer
 * updates that it sends. The SDK shall call the
 * VNCViewerHSMLInformationCallback() each time an HSML information rectangle
 * is received.
 *
 * When receiving the VNCViewerHSMLInformationCallback(), the viewer
 * application must immediately establish an HSML connection with the
 * MirrorLink server. This must be done separately. The VNC HSML SDK may be
 * used to accomplish this. The viewer application should match the HSML source
 * identifer given in the VNCViewerHSMLInformationCallback() with that reported
 * in the corresponding HSML session.
 *
 * Both the HSML and VNC sessions must then run concurrently. If the HSML
 * session is lost, then the VNC session should also be terminated. Likewise,
 * the HSML session should be terminated if the VNC session is lost.
 *
 * \section mirrorlink_references References
 *
 *  -# \anchor ml_vnc [ML-VNC] CCC-TS-010 VNC Based Display and Control,
 *     versions 1.1, 1.2 and 1.3
 *  -# \anchor ml_hsml [ML-HSML] CCC-TS-054 High Speed Media Link (HSML),
 *     version 1.2 and 1.3
 */

#include "mirrorlinkkeysymdef.h"
#include "vnctypes.h"
#include "vncmirrorlinktypes.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Constants for use with the framebufferConfiguration field in the
 * VNCServerDisplayConfiguration and VNCClientDisplayConfiguration
 * structures.
 *
 * If any of these bits is set in the VNCServerDisplayConfiguration, then the
 * server is notifying the viewer application that the server supports the
 * feature.
 *
 * If you set any of these bits in the VNCClientDisplayConfiguration, then the
 * viewer is indicating to the server that it intends to make use of the
 * feature.
 *
 * \see VNCServerDisplayConfiguration, VNCClientDisplayConfiguration,
 * VNCDeviceStatusRequest,
 * VNCViewerServerDisplayConfigurationCallback
 */
typedef enum
{
    /**
     * The server is capable of switching orientations, as instructed by an
     * appropriate DeviceStatusRequest message.
     *
     * This capability has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this capability if it supports
     * MirrorLink 1.3 or above.
     */
    VNCFramebufferConfigurationServerSideOrientationSwitch        = 0x0001,

    /**
     * The server is capable of rotating its framebuffer, as instructed by an
     * appropriate DeviceStatusRequest message.
     *
     * This capability has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this capability if it supports
     * MirrorLink 1.3 or above.
     */
    VNCFramebufferConfigurationServerSideRotation                 = 0x0002,

    /**
     * The server is capable of upscaling its framebuffer if its own dimensions
     * are less than those of the client display, which are specified in the
     * VNCClientDisplayConfiguration structure.
     *
     * This capability has been deprecated in MirrorLink 1.3 for servers. The
     * viewer application may still advertise this capability if it supports
     * MirrorLink 1.3 or above.
     */
    VNCFramebufferConfigurationUpScaling                          = 0x0004,

    /**
     * The server is capable of downscaling its framebuffer if its own
     * dimensions are greater than those of the client display, which are
     * specified in the VNCClientDisplayConfiguration structure.
     *
     * This capability has been deprecated in MirrorLink 1.3 for viewers. The
     * viewer application should not advertise this capability if it supports
     * MirrorLink 1.3 or above.
     */
    VNCFramebufferConfigurationDownScaling                        = 0x0008,

    /**
     * The server is capable of suppressing empty update rectangles, so that
     * the viewer application does not have to deal with them.
     *
     * The Viewer SDK always detects and discards empty update rectangles. This
     * capability has been removed in MirrorLink 1.1 and above. The viewer
     * application should therefore not set this flag in the
     * ClientDisplayConfiguration message.
     */
    VNCFramebufferConfigurationReplaceEmptyUpdates                = 0x0010,

    /**
     * The server supports FramebufferAlternativeText messages.
     *
     * The Viewer SDK does not support FramebufferAlternativeText messages.
     * This feature has also been deprecated in MirrorLink 1.3. The viewer
     * application should therefore not set this flag in the
     * ClientDisplayConfiguration message.
     */
    VNCFramebufferConfigurationSupportsFramebufferAlternativeText = 0x0020,
} VNCFramebufferConfiguration;

/**
 * \brief Constants for use with the pixelFormatSupport fields in the
 * VNCServerDisplayConfiguration and VNCClientDisplayConfiguration structures.
 *
 * In a MirrorLink session, care should be taken to only request a pixel
 * format that the server has said that it supports.  (Support for 32-bit
 * ARGB888 and 16-bit RGB565 is mandatory.)
 *
 * Note that the Viewer SDK does not support any grayscale pixel formats and
 * does not support 24-bit true color pixel formats.  (In this context, 24-bit
 * is the size of each pixel, as given by the bitsPerPixel field in the
 * VNCPixelFormat structure, and not the color depth.) Additionally, big-endian
 * pixel formats are not supported in MirrorLink sessions.
 *
 * \see VNCServerDisplayConfiguration, VNCClientDisplayConfiguration,
 * VNCViewerSetPixelFormat, VNCPixelFormat
 */
typedef enum
{
    /**
     * Supports no pixel formats. For use only in the
     * VNCClientDisplayConfiguration message, where it indicates that the
     * transform encoding implementation cannot transcode between pixel
     * formats.
     *
     * The viewer application should advertise support for either
     * VNCPixelFormatSupportARGB888_32, VNCPixelFormatSupportRGB565_16, or
     * both, in the VNCClientDisplayConfiguration structure if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportNone         = 0x00000000,
    /**
     * Supports the 32-bit ARGB888 pixel format.
     *
     * The viewer application should advertise support for either this pixel
     * format, VNCPixelFormatSupportRGB565_16, or both, in the
     * VNCClientDisplayConfiguration structure if it supports MirrorLink 1.3
     * or above.
     */
    VNCPixelFormatSupportARGB888_32   = 0x00000001,
    /**
     * Supports any other 32-bit true color pixel formats. For use only in the
     * VNCServerDisplayConfiguration message.
     *
     * These pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportAny_32       = 0x00000080,
    /**
     * Supports the RGB888 pixel format.
     *
     * This pixel format has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this pixel format if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportRGB888_24    = 0x00000100,
    /**
     * Supports any other 24-bit true color pixel formats. For use only in the
     * VNCServerDisplayConfiguration message.
     *
     * These pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportAny_24       = 0x00008000,
    /**
     * Supports the RGB565 pixel formats.
     *
     * The viewer application should advertise support for either this pixel
     * format, VNCPixelFormatSupportARGB888_32, or both, in the
     * VNCClientDisplayConfiguration structure if it supports MirrorLink 1.3
     * or above.
     */
    VNCPixelFormatSupportRGB565_16    = 0x00010000,
    /**
     * Supports the RGB555 pixel formats.
     *
     * These pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportRGB555_16    = 0x00020000,
    /**
     * Supports the RGB444 pixel formats.
     *
     * These pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportRGB444_16    = 0x00040000,
    /**
     * Supports the RGB343 pixel formats.
     *
     * These pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportRGB343_16    = 0x00080000,
    /**
     * Supports any other 16-bit true color pixel formats.
     *
     * These pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportAny_16       = 0x00800000,
    /**
     * Supports 16-bit grayscale.
     *
     * This pixel format has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this pixel format if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportGrayscale_16 = 0x01000000,
    /**
     * Supports 8-bit grayscale.
     *
     * This pixel format has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this pixel format if it supports
     * MirrorLink 1.3 or above.
     */
    VNCPixelFormatSupportGrayscale_8  = 0x02000000,
} VNCPixelFormatSupport;

/**
 * \brief Constants for use with the resizeFactors fields in the
 * VNCClientDisplayConfiguration structure.
 *
 * The Transform encoding has been deprecated in MirrorLink 1.3. The viewer
 * application should not advertise support for any resize factors in
 * MirrorLink 1.3 sessions and above.
 *
 * \see VNCClientDisplayConfiguration
 */
typedef enum
{
    /**
     * No supported resize factors.
     *
     * In MirrorLink 1.3 sessions and above, only this value should be used.
     * This value should not be used in MirrorLink 1.2 sessions and below.
     */
    VNCResizeFactorNone = 0x00000000,

    /** Resize factor of 1/1. */
    VNCResizeFactor1_1  = 0x00000001,
    /** Resize factor of 1/2. */
    VNCResizeFactor1_2  = 0x00000002,
    /** Resize factor of 1/3. */
    VNCResizeFactor1_3  = 0x00000004,
    /** Resize factor of 1/4. */
    VNCResizeFactor1_4  = 0x00000008,
    /** Resize factor of 1/5. */
    VNCResizeFactor1_5  = 0x00000010,
    /** Resize factor of 1/6. */
    VNCResizeFactor1_6  = 0x00000020,
    /** Resize factor of 1/8. */
    VNCResizeFactor1_8  = 0x00000040,
    /** Resize factor of 1/10. */
    VNCResizeFactor1_10 = 0x00000080,
    /** Resize factor of 1/16. */
    VNCResizeFactor1_16 = 0x00000100,
    /** Resize factor of 1/32. */
    VNCResizeFactor1_32 = 0x00000200,
    /** Resize factor of 2/3. */
    VNCResizeFactor2_3  = 0x00000400,
    /** Resize factor of 3/4. */
    VNCResizeFactor3_4  = 0x00000800,
} VNCResizeFactor;

/**
 * \brief Constants for use with the miscKeySupport field in the 
 * VNCServerEventConfiguration and \ref VNCClientEventConfiguration structures.
 *
 * In the VNCServerEventConfiguration structure, these values are combined to
 * indicate to the viewer application which input events and features the
 * server supports.
 *
 * In the \ref VNCClientEventConfiguration structure, these values are combined
 * to indicate to the server which events and features the viewer application
 * will use.
 *
 * Further details on the meaning of these keys can be found in
 * mirrorlinkkeysymdef.h.
 *
 * \see VNCServerEventConfiguration, VNCClientEventConfiguration
 */
typedef enum
{
    /**
     * Supports ITU keypad events.
     *
     * These events have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to support these events if it supports
     * MirrorLink 1.3 or above.
     */
    VNCMiscKeySupportITU                    = 0x00000001,

    /**
     * Supports the Virtual Keyboard Trigger feature.
     *
     * This feature has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this feature if it supports
     * MirrorLink 1.3 or above.
     *
     * \see VNCViewerSendVirtualKeyboardTriggerRequest
     * \see VNCViewerVirtualKeyboardTriggerCallback
     */
    VNCMiscKeySupportVirtualKeyboardTrigger = 0x00000002,

    /**
     * Supports the Key Event Listing feature.
     *
     * This feature has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this feature if it supports
     * MirrorLink 1.3 or above.
     *
     * \see VNCViewerSendKeyEventListingRequest
     * \see VNCViewerKeyEventListingCallback
     */
    VNCMiscKeySupportKeyEventListing        = 0x00000004,

    /**
     * Supports the Event Mapping feature.
     *
     * This feature has been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use this feature if it supports
     * MirrorLink 1.3 or above.
     *
     * \see VNCViewerSendEventMappingRequest
     * \see VNCViewerEventMappingCallback
     */
    VNCMiscKeySupportEventMapping           = 0x00000008,

    /**
     * Supports Function Key 0.
     */
    VNCMiscKeySupportFunctionKey0           = 0x00000100,

    /**
     * Supports Function Key 1.
     */
    VNCMiscKeySupportFunctionKey1           = 0x00000200,

    /**
     * Supports Function Key 2.
     */
    VNCMiscKeySupportFunctionKey2           = 0x00000400,

    /**
     * Supports Function Key 3.
     */
    VNCMiscKeySupportFunctionKey3           = 0x00000800,

    /**
     * Supports Function Key 4.
     */
    VNCMiscKeySupportFunctionKey4           = 0x00001000,

    /**
     * Supports Function Key 5.
     */
    VNCMiscKeySupportFunctionKey5           = 0x00002000,

    /**
     * Supports Function Key 6.
     */
    VNCMiscKeySupportFunctionKey6           = 0x00004000,

    /**
     * Supports Function Key 7.
     */
    VNCMiscKeySupportFunctionKey7           = 0x00008000,

} VNCMiscKeySupport;

/**
 * \brief Constants for use with the pointerSupport field in the 
 * VNCServerEventConfiguration and \ref VNCClientEventConfiguration structures.
 *
 * In the VNCServerEventConfiguration structure, these values are combined to
 * indicate to the viewer application which input events the server supports.
 *
 * In the \ref VNCClientEventConfiguration structure, these values are combined
 * to indicate to the server which events the viewer application will generate.
 *
 * \see VNCServerEventConfiguration, VNCClientEventConfiguration
 */
typedef enum
{
    /** \cond */
    VNCPointerSupportPointerEvents               = 0x00000001,
    VNCPointerSupportTouchEvents                 = 0x00000002,
    VNCPointerSupportPointerButton1              = 0x00000100,
    VNCPointerSupportPointerButton2              = 0x00000200,
    VNCPointerSupportPointerButton3              = 0x00000400,
    VNCPointerSupportPointerButton4              = 0x00000800,
    VNCPointerSupportPointerButton5              = 0x00001000,
    VNCPointerSupportPointerButton6              = 0x00002000,
    VNCPointerSupportPointerButton7              = 0x00004000,
    VNCPointerSupportPointerButton8              = 0x00008000,
    VNCPointerSupportPointerButtonMASK           = 0x0000FF00,
    VNCPointerSupportTouchCount1                 = 0x00000000,
    VNCPointerSupportTouchCount2                 = 0x00010000,
    VNCPointerSupportTouchCount3                 = 0x00020000,
    VNCPointerSupportTouchCount4                 = 0x00030000,
    VNCPointerSupportTouchCount5                 = 0x00040000,
    VNCPointerSupportTouchCount6                 = 0x00050000,
    VNCPointerSupportTouchCount7                 = 0x00060000,
    VNCPointerSupportTouchCount8                 = 0x00070000,
    VNCPointerSupportTouchCount9                 = 0x00080000,
    VNCPointerSupportTouchCount10                = 0x00090000,
    VNCPointerSupportTouchCountMASK              = 0x00FF0000,
    VNCPointerSupportTouchEventPressureMaskSHIFT = 24
    /** \endcond */
} VNCPointerSupport;

/**
 * \brief Structure holding a decoded ServerDisplayConfiguration MirrorLink
 * extension message that has been received from the server.
 *
 * \see VNCViewerServerDisplayConfigurationCallback,
 * VNCFramebufferConfiguration, VNCPixelFormatSupport,
 * VNCClientDisplayConfiguration
 */
typedef struct
{
    /** 
     * The major version number of the MirrorLink specification followed by
     * the server.
     */
    vnc_uint8_t serverMajorVersion;
    /** 
     * The minor version number of the MirrorLink specification followed by
     * the server.
     */
    vnc_uint8_t serverMinorVersion;
    /** 
     * A combination of \ref VNCFramebufferConfiguration values describing the
     * server's capabilities with respect to its framebuffer.
     *
     * Some capabilities have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these capabilities if it supports
     * MirrorLink 1.3 or above. See \ref VNCFramebufferConfiguration for
     * further information.
     */
    vnc_uint16_t framebufferConfiguration;
    /** 
     * The width of each pixel in the server display relative to its height.
     *
     * This has no effect on the Viewer SDK, but you may wish to take note of
     * it when displaying the framebuffer to the user.
     *
     * This field was deprecated in MirrorLink 1.1. The viewer application
     * should disregard this value if it supports MirrorLink 1.1 or above. A
     * relative pixel width value of 1 should be assumed.
     */
    vnc_uint16_t relativePixelWidth;
    /** 
     * The height of each pixel in the server display relative to its width.
     *
     * This has no effect on the Viewer SDK, but you may wish to take note of
     * it when displaying the framebuffer to the user.
     *
     * This field was deprecated in MirrorLink 1.1. The viewer application
     * should disregard this value if it supports MirrorLink 1.1 or above. A
     * relative pixel height value of 1 should be assumed.
     */
    vnc_uint16_t relativePixelHeight;
    /** 
     * A combination of \ref VNCPixelFormatSupport values that detail the 
     * server's capabilities with respect to its framebuffer.
     *
     * Some pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above. See \ref VNCPixelFormatSupport for further
     * information.
     */
    vnc_uint32_t pixelFormatSupport;
} VNCServerDisplayConfiguration;

/**
 * \brief Structure holding a ClientDisplayConfiguration MirrorLink extension
 * message to be sent to the server.
 *
 * The SDK pre-fills the VNCClientDisplayConfiguration structure with sensible
 * default values before sending it to the server.  A viewer application is
 * free to use its VNCViewerServerDisplayConfigurationCallback to
 * override these values, but it is the viewer application's responsibility to
 * ensure that the overridden values are sane.
 *
 * \see VNCViewerServerDisplayConfigurationCallback,
 * VNCFramebufferConfiguration, VNCPixelFormatSupport, VNCResizeFactor,
 * VNCClientDisplayConfiguration
 */
typedef struct
{
    /** 
     * The major version number of the MirrorLink specification followed by
     * the client.  This may not exceed the major version provided by the
     * server.
     *
     * Pre-filled to the major version advertised by the server. If the version
     * advertised by the server is newer than the version supported by the SDK,
     * this will be limited to the maximum SDK-supported version.
     */
    vnc_uint8_t clientMajorVersion;
    /** 
     * The minor version number of the MirrorLink specification followed by
     * the client.  This may not exceed the minor version provided by the
     * server.
     *
     * Pre-filled to the minor version advertised by the server. If the version
     * advertised by the server is newer than the version supported by the SDK,
     * this will be limited to the maximum SDK-supported version.
     */
    vnc_uint8_t clientMinorVersion;
    /** 
     * A combination of \ref VNCFramebufferConfiguration values describing
     * which of the server's advertised framebuffer capabilities it intends to
     * use.
     *
     * Pre-filled to
     * (VNCFramebufferConfigurationUpScaling |
     *  VNCFramebufferConfigurationDownScaling),
     * with the addition of 
     * VNCFramebufferConfigurationServerSideOrientationSwitch if the server has
     * advertised that it supports it.
     *
     * Some capabilities have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these capabilities if it supports
     * MirrorLink 1.3 or above. See \ref VNCFramebufferConfiguration for
     * further information.
     */
    vnc_uint16_t framebufferConfiguration;
    /** 
     * The width in pixels of the area in which the viewer application will
     * display the framebuffer.
     *
     * Pre-filled to the width of the server display in pixels.  However, you
     * should always override this with the actual width of the viewer-side
     * display, if at all possible.
     *
     * Note that MirrorLink 1.3 requires a minimum of 800 for this value.
     */
    vnc_uint16_t clientDisplayWidthPixels;
    /** 
     * The height in pixels of the area in which the viewer application will
     * display the framebuffer.
     *
     * Pre-filled to the height of the server display in pixels.  However, you
     * should always override this with the actual height of the viewer-side
     * display, if at all possible.
     *
     * Note that MirrorLink 1.3 requires a minimum of 480 for this value.
     */
    vnc_uint16_t clientDisplayHeightPixels;
    /**
     * The width in millimeters of the area in which the viewer application
     * will display the framebuffer.
     *
     * Pre-filled to 0.
     */
    vnc_uint16_t clientDisplayWidthMillimeters;
    /**
     * The height in millimeters of the area in which the viewer application
     * will display the framebuffer.
     *
     * Pre-filled to 0.
     */
    vnc_uint16_t clientDisplayHeightMillimeters;
    /**
     * The expected distance between the viewer display and the user in
     * millimeters.
     *
     * In-car applications may be able to provide the server with a good
     * estimate for this value.
     *
     * Pre-filled to 0.
     */
    vnc_uint16_t clientDistanceFromUserMillimeters;

    /** 
     * A combination of \ref VNCPixelFormatSupport values that detail the 
     * pixel formats that are supported by the client.
     *
     * Some pixel formats have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these pixel formats if it supports
     * MirrorLink 1.3 or above. See \ref VNCPixelFormatSupport for further
     * information.
     */
    vnc_uint32_t pixelFormatSupport;

    /** 
     * A combination of \ref VNCResizeFactor values that detail the resize
     * factors that are supported by the client for use with the Transform
     * encoding.
     *
     * The Transform encoding has been deprecated in MirrorLink 1.3. The viewer
     * application should not advertise support for any resize factors in
     * MirrorLink 1.3 sessions or above. See \ref VNCResizeFactor for further
     * information.
     */
    vnc_uint32_t resizeFactors;
} VNCClientDisplayConfiguration;

/**
 * \brief Structure holding a decoded ServerEventConfiguration MirrorLink
 * extension message that has been received from the server.
 *
 * A VNCServerEventConfiguration structure is supplied to the optional 
 * \ref VNCViewerServerEventConfigurationCallback.
 *
 * \see VNCViewerServerEventConfigurationCallback,
 * VNCKnobKeySupport, VNCDeviceKeySupport, VNCMultimediaKeySupport,
 * VNCMiscKeySupport, VNCPointerSupport
 */
typedef struct
{
    /**
     * ISO 639-1 language code for the server's keyboard layout.
     *
     * The SDK normalizes the case of this field so that the value is always
     * lowercase.
     */
    char keyboardLanguage[3];
    /**
     * ISO 3166-1 country code for the server's keyboard layout.
     *
     * For example, if the server has an American keyboard layout, then
     * keyboardLanguage will be 'en' and keyboardCountry will be 'US'.
     *
     * The SDK normalizes the case of this field so that the value is always
     * uppercase.
     */
    char keyboardCountry[3];
    /**
     * ISO 639-1 language code for the server's user interface language.
     *
     * The SDK normalizes the case of this field so that the value is always
     * lowercase.
     */
    char uiLanguage[3];
    /**
     * ISO 3166-1 country code for the server's user interface language.
     *
     * For example, if the server's user interface is US English, then
     * uiLanguage will be 'en' and uiCountry will be 'US'.
     *
     * The SDK normalizes the case of this field so that the value is always
     * uppercase.
     */
    char uiCountry[3];
    /**
     * Indicates the server's support for knob key input.
     *
     * The value is a combination of the values in the \ref VNCKnobKeySupport
     * enumeration.  For example, if the VNCKnobKeySupport0PushZ bit is set,
     * then the server will respond to KeyEvents of type
     * XK_Knob_2D_shift_push(0).
     */
    vnc_uint32_t knobKeySupport;
    /**
     * Indicates the server's support for MirrorLink device key input.
     *
     * The value is a combination of the values in the \ref VNCDeviceKeySupport
     * enumeration.  For example, if the VNCDeviceKeySupportSoftLeft bit is
     * set, then the server will respond to KeyEvents of type
     * XK_Device_Soft_left.
     */
    vnc_uint32_t deviceKeySupport;
    /**
     * Indicates the server's support for multimedia key input.
     *
     * The value is a combination of the values in the
     * \ref VNCMultimediaKeySupport enumeration.  For example, if the
     * VNCMultimediaKeySupportPlay bit is set, then the server will respond to
     * KeyEvents of type XK_Multimedia_Play.
     */
    vnc_uint32_t multimediaKeySupport;
    /**
     * Indicates the server's support for miscellaneous MirrorLink key
     * input.
     *
     * The value is a combination of the values in the \ref VNCMiscKeySupport
     * enumeration.  For example, if the VNCMiscKeySupportITU bit is
     * set, then the server will respond to KeyEvents with ITU keypad key
     * symbols (XK_ITU_Key_0 etc).
     *
     * Some features have been deprecated in MirrorLink 1.3. The viewer
     * application should not attempt to use these features if it supports
     * MirrorLink 1.3 or above. See \ref VNCMiscKeySupport for further
     * information.
     */
    vnc_uint32_t miscKeySupport;
    /**
     * Indicates the server's support for pointer / touchscreen input.
     *
     * The value is a combination of the values in the \ref VNCPointerSupport
     * enumeration.  For example, if the VNCPointerSupportPointerButton1 bit
     * is set, then the server will respond to PointerEvents with the
     * VNCPointerDeviceButtonLeft bit set.
     */
    vnc_uint32_t pointerSupport;
} VNCServerEventConfiguration;

/**
 * \brief Structure holding a ClientEventConfiguration MirrorLink
 * extension message to be sent to the server.
 *
 * This structure is identical to the VNCServerEventConfiguration structure,
 * except that the fields indicate which of the server's advertised
 * features the client wishes to use.
 *
 * The SDK pre-fills the VNCClientEventConfiguration structure with sensible
 * default values before sending it to the server.  A viewer application is
 * free to use its VNCViewerServerEventConfigurationCallback to
 * override these values, but it is the viewer application's responsibility to
 * ensure that the overridden values are sane.
 *
 * The values that the SDK pre-fills are:
 *
 *  - keyboardLanguage - identical to keyboardLanguage in the
 *    VNCServerEventConfiguration.
 *  - keyboardCountry - identical to keyboardLanguage in the
 *    VNCServerEventConfiguration.
 *  - uiLanguage - identical to uiLanguage in the VNCServerEventConfiguration
 *  - uiCountry - identical to uiLanguage in the VNCServerEventConfiguration.
 *  - knobKeySupport - (VNCKnobKeySupport0ShiftX | VNCKnobKeySupport0ShiftY),
 *    masked by the knobKeySupport value from the VNCServerEventConfiguration.
 *    (Support for these features of a single 2-D knob is equivalent to support
 *    for cursor keys.)
 *  - multimediaKeySupport - zero.
 *  - deviceKeySupport - VNCDeviceKeySupportALL, masked by the deviceKeySupport
 *    value from the VNCServerEventConfiguration.
 *  - miscKeySupport - VNCMiscKeySupportITU, masked by the miscKeySupport value
 *    from the VNCServerEventConfiguration. Applies to MirrorLink 1.2 sessions
 *    and below only.
 *  - pointerSupport - (VNCPointerSupportPointerEvents |
 *                      VNCPointerSupportPointerButton1 |
 *                      VNCPointerSupportPointerButton2 |
 *                      VNCPointerSupportPointerButton3 |
 *                      VNCPointerSupportPointerButton4 |
 *                      VNCPointerSupportPointerButton5 |
 *                      VNCPointerSupportPointerButton6 |
 *                      VNCPointerSupportPointerButton7 |
 *                      VNCPointerSupportPointerButton8),
 *    masked by the pointerSupport value from the VNCServerEventConfiguration.
 *
 * If the viewer application supports MirrorLink 1.1 or above, and does not
 * support pointer events, then it is required to support certain knob key
 * events. The viewer application should set the following knob key support
 * bits in the event configuration:
 *
 *  - VNCKnobKeySupport0ShiftX
 *  - VNCKnobKeySupport0ShiftY
 *  - VNCKnobKeySupport0PushZ
 *  - VNCKnobKeySupport0RotateZ
 *
 * From MirrorLink 1.3 onwards, viewer applications not supporting pointer
 * events must additionally support the back device key, and so should also set
 * the VNCDeviceKeySupportBackward device key support bit in the event
 * configuration.
 *
 * Some miscellaneous key support features have been deprecated in MirrorLink
 * 1.3. The viewer application should not attempt to use these features if it
 * supports MirrorLink 1.3 or above. See \ref VNCMiscKeySupport for further
 * information.
 *
 * \see VNCViewerServerEventConfigurationCallback,
 * VNCKnobKeySupport, VNCDeviceKeySupport, VNCMultimediaKeySupport,
 * VNCMiscKeySupport, VNCPointerSupport
 */
typedef VNCServerEventConfiguration VNCClientEventConfiguration;

/**
 * \brief Structure holding a DeviceStatusRequest MirrorLink extension
 * message to be sent to the server.
 *
 * The SDK always sends at least one DeviceStatusRequest to the server,
 * immediately after the handshake finishes.  This initial
 * VNCDeviceStatusRequest structure is passed to the
 * VNCViewerServerEventConfigurationCallback, so that the
 * application may alter it if required.
 *
 * The initial VNCDeviceStatusRequest structure is pre-filled by the SDK as
 * follows in MirrorLink 1.2 sessions and below:
 *
 *  - features - (VNCDeviceStatusFeatureDeviceLockDisabled |
 *                VNCDeviceStatusFeatureScreenSaverDisabled |
 *                VNCDeviceStatusFeatureNightModeDisabled |
 *                VNCDeviceStatusFeatureVoiceInputDisabled |
 *                VNCDeviceStatusFeatureVoiceInputReroutingDisabled |
 *                VNCDeviceStatusFeatureDriverDistractionAvoidanceDisabled |
 *                VNCDeviceStatusFeatureFramebufferRotation0Degrees |
 *                VNCDeviceStatusFeatureOrientationLandscape)
 *
 * Some device features have been deprecated in MirrorLink 1.3. Therefore, the
 * structure is pre-filled as follows in MirrorLink 1.3 sessions and above:
 *
 *  - features - (VNCDeviceStatusFeatureNightModeDisabled |
 *                VNCDeviceStatusFeatureVoiceInputDisabled |
 *                VNCDeviceStatusFeatureVoiceInputReroutingDisabled |
 *                VNCDeviceStatusFeatureDriverDistractionAvoidanceDisabled |
 *                VNCDeviceStatusFeatureFramebufferRotation0Degrees |
 *                VNCDeviceStatusFeatureOrientationLandscape)
 *
 * After this initial DeviceStatusRequest, the viewer application may choose to
 * send additional DeviceStatusRequest messages to the server.  To do this,
 * call VNCViewerSendDeviceStatusRequest.
 *
 * \see VNCViewerDeviceStatusCallback,
 * VNCViewerServerEventConfigurationCallback,
 * VNCViewerSendDeviceStatusRequest, VNCDeviceStatus,
 * VNCDeviceStatusFeature
 */
typedef VNCDeviceStatus VNCDeviceStatusRequest;

/**
 * \brief Structure holding a FramebufferBlockingNotification MirrorLink
 * extension message to be sent to the server.
 *
 * To send a FramebufferBlockingNotification, the viewer application should
 * initialize the structure and then call
 * \ref VNCViewerSendFramebufferBlockingNotification.
 *
 * \see VNCViewerSendFramebufferBlockingNotification,
 * VNCFramebufferBlockReason, VNCViewerContextInformationCallback
 */
typedef struct
{
    /**
     * The unique ID of the application that has been blocked.
     *
     * The value should be taken from the most recent VNCContextInformation
     * passed to the viewer application for the relevant part of the
     * framebuffer.
     */
    vnc_uint32_t applicationUniqueId;
    /**
     * The reason for the viewer application's decision to block the relevant
     * part of the framebuffer.
     *
     * The value should be a combination of values from the
     * \ref VNCFramebufferBlockReason.  Depending on the reason, the server may
     * choose to dismiss the application in question.
     *
     * Some framebuffer blocking reasons have been deprecated in MirrorLink
     * 1.3. The viewer application should not specify these reasons if it
     * supports MirrorLink 1.3 or above. See \ref VNCFramebufferBlockReason for
     * further information.
     */
    VNCFramebufferBlockReason blockReason;
} VNCFramebufferBlockingNotification;

/**
 * \brief Structure holding an AudioBlockingNotification MirrorLink
 * extension message to be sent to the server.
 *
 * To send an AudioBlockingNotification, the viewer application should
 * initialize the structure and then call
 * VNCViewerSendAudioBlockingNotification.
 *
 * \see VNCViewerSendAudioBlockingNotification, VNCAudioBlockReason
 */
typedef struct
{
    /**
     * The unique ID of the application that has been blocked.
     *
     * The value should be taken from the audio negotiation with the MirrorLink
     * UPnP server.
     */
    vnc_uint32_t applicationUniqueId;
    /**
     * The reason for the viewer application's decision to block the audio.
     *
     * The value should be a combination of values from the
     * VNCAudioBlockReason.  Depending on the reason, the server may choose to
     * suspend playback of the audio stream.
     *
     * A value of VNCAudioBlockReasonUnblock indicates that the server should
     * resume playback of the audio stream.
     *
     * Some audio blocking reasons have been deprecated in MirrorLink 1.3. The
     * viewer application should not specify these reasons if it supports
     * MirrorLink 1.3 or above. See \ref VNCAudioBlockReason for further
     * information.
     */
    VNCAudioBlockReason blockReason;
} VNCAudioBlockingNotification;

/**
 * \brief Structure holding a decoded Context Information rectangle that has
 * been received from the server.
 *
 * \see VNCViewerContextInformationCallback
 */
typedef struct
{
    /**
     * The unique ID of the application that has drawn to the relevant part of
     * the framebuffer.
     */
    vnc_uint32_t applicationUniqueId;
    /**
     * The server's level of trust that the information in the
     * applicationCategory field is correct.
     */
    VNCTrustLevel applicationCategoryTrustLevel;
    /**
     * The server's level of trust that the information in the contentCategory
     * field is correct.
     */
    VNCTrustLevel contentCategoryTrustLevel;
    /**
     * The category and sub-category into which the application falls.
     */
    VNCApplicationCategory applicationCategory;
    /**
     * The category of the content that the application is presenting.
     */
    VNCVisualContentCategory contentCategory;
    /**
     * A bit-field of the content rules, negotiated in the MirrorLink UPnP
	 * stream, with which the application has complied.
     */
    vnc_uint32_t contentRulesFollowed;
} VNCContextInformation;

/**
 * \brief Describes the type of virtual keyboard being requested.
 *
 * Virtual keyboard trigger support has been deprecated in MirrorLink 1.3.
 * This structure is not used in MirrorLink 1.3 sessions and above.
 *
 * For further information in MirrorLink 1.2 and below, see section 5.6,
 * 'Virtual Keyboard Trigger Messages', in \ref ml_vnc "[ML-VNC]".
 *
 * \see VNCVirtualKeyboardTrigger
 */
typedef enum
{
  /** The validity of the cursor position in the message. */
  VNCVirtualKeyboardTriggerFlagCursorValid            = 0x00000001,
  /** The validity of the input area in the message. */
  VNCVirtualKeyboardTriggerFlagAreaValid              = 0x00000002,
  /** If a key event listing follows the message. */
  VNCVirtualKeyboardTriggerFlagKeyEventListingFollows = 0x00000004,
  /** If the virtual keyboard should be removed for display. */
  VNCVirtualKeyboardTriggerFlagRemoveKeyboard         = 0x00000008,
  /** If text entry exchange is supported. */
  VNCVirtualKeyboardTriggerFlagTextEntryExchange      = 0x00000010,
  /** Unknown virtual keyboard type. */
  VNCVirtualKeyboardTriggerFlagTypeUnknown            = 0x00000000,
  /** QWERTY virtual keyboard type*/
  VNCVirtualKeyboardTriggerFlagTypeQwerty             = 0x00000100,
  /** Numeric keyboard type (including '+' and '#'). */
  VNCVirtualKeyboardTriggerFlagTypeNumeric            = 0x00000200,
  /** Bitmask to use before comparing with a keyboard type.*/
  VNCVirtualKeyboardTriggerFlagTypeMASK               = 0x0000FF00,
} VNCVirtualKeyboardTriggerFlags;

/**
 * \brief Structure holding a decoded virtual keyboard trigger message
 * that has been received from the server.
 *
 * Virtual keyboard trigger support has been deprecated in MirrorLink 1.3.
 * This structure is not used in MirrorLink 1.3 sessions and above.
 *
 * \see VNCViewerVirtualKeyboardTriggerCallback
 */
typedef struct
{
  /**
   * A bit-field representing describing the requested virtual keyboard.
   *
   * The value should be a combination of values from the
   * \ref VNCVirtualKeyboardTriggerFlags enumeration.
   *
   */
  VNCVirtualKeyboardTriggerFlags flags;
  /**
   * Describes the cursor location for the virtual keyboard request.
   *
   * Only valid if \ref VNCVirtualKeyboardTriggerFlagCursorValid is
   * set in the flags field.
   *
   * \see VNCVirtualKeyboardTriggerFlagCursorValid
   */
  VNCPoint cursorPosition;
  /**
   * Describes the text input area for the virtual keyboard request.
   *
   * Only valid if \ref VNCVirtualKeyboardTriggerFlagAreaValid is
   * set in the flags field.
   *
   * \see VNCVirtualKeyboardTriggerFlagAreaValid
   */
  VNCRectangle textInputArea;
} VNCVirtualKeyboardTrigger;

/**
 * \brief Describes the supported virtual keyboard being requested.
 *
 * Virtual keyboard trigger support has been deprecated in MirrorLink 1.3.
 * This structure is not used in MirrorLink 1.3 sessions and above.
 *
 * For further information in MirrorLink 1.2 and below, see section 5.6,
 * 'Virtual Keyboard Trigger Messages', in \ref ml_vnc "[ML-VNC]".
 *
 * \see VNCVirtualKeyboardTriggerRequest
 */
typedef enum
{
  /** Enable the trigger. */
  VNCVirtualKeyboardTriggerRequestFlagEnable            = 0x00000001,
  /** Supports text entry exchange. */
  VNCVirtualKeyboardTriggerRequestFlagTextEntryExchange = 0x00000002,
  /** Mask to use to set the maximum characters in a text entry exchange. */
  VNCVirtualKeyboardTriggerRequestFlagMaxCharactersMASK = 0x0000FF00,
} VNCVirtualKeyboardTriggerRequestFlags;

/**
 * \brief Structure holding a VirtualKeyboardTriggerRequest MirrorLink
 * extension message to be sent to the server.
 *
 * To send a VirtualKeyboardTriggerRequest, the viewer application
 * should initialize the structure and then call
 * VNCViewerSendVirtualKeyboardTriggerRequest.
 *
 * Virtual keyboard trigger support has been deprecated in MirrorLink 1.3.
 * This structure is not used in MirrorLink 1.3 sessions and above.
 *
 * \see VNCViewerSendVirtualKeyboardTriggerRequest
 */
typedef struct
{
  /**
   * A bit-field representing describing the requested virtual keyboard.
   *
   * The value should be a combination of values from the
   * \ref VNCVirtualKeyboardTriggerRequestFlags enumeration.
   */
  VNCVirtualKeyboardTriggerRequestFlags flags;
} VNCVirtualKeyboardTriggerRequest;

/**
 * Used in an EventMapping message to indicate no mapping exists.
 *
 * \see VNCEventMapping
 */
#define VNCEventMappingNotMapped 0

/**
 * Describes an EventMapping currently in place on the server.
 *
 * For further information, see section 5.4, 'Event Mapping Messages',
 * in \ref ml_vnc "[ML-VNC]".
 *
 * \see VNCViewerEventMappingCallback
 */
typedef struct
{
  /**
   * The client key symbol value that this mapping describes.
   */
  vnc_uint32_t clientKeySymbol;
  /**
   * The server key symbol value for this mapping.
   *
   * VNCEventMappingNotMapped if the symbol value is not mapped.
   */
  vnc_uint32_t serverKeySymbol;
} VNCEventMapping;

/**
 * Used in an EventMappingRequest to query the current event
 * mapping from the server.
 *
 * For further information, see section 5.4, 'Event Mapping Messages',
 * in \ref ml_vnc "[ML-VNC]".
 *
 * \see VNCEventMappingRequest
 */
#define VNCEventMappingRequestQuery 0

/**
 * \brief Structure holding an EventMappingRequest MirrorLink
 * extension message to be sent to the server.
 *
 * To send an EventMappingRequest, the viewer application
 * should initialize the structure and then call
 * VNCViewerSendEventMappingRequest.
 *
 * \see VNCViewerSendEventMappingRequest
 */
typedef struct
{
  /**
   * The client key symbol value that this mapping describes.
   */
  vnc_uint32_t clientKeySymbol;
  /**
   * The server key symbol value for this mapping.
   *
   * VNCEventMappingRequestQuery if the viewer application is requesting the
   * current server mapping.
   */
  vnc_uint32_t serverKeySymbol;
} VNCEventMappingRequest;

/**
 * Describes the current list of valid key symbols advertised by the server.
 *
 * Key event listing support is deprecated in MirrorLink 1.3.
 * This structure is not used in MirrorLink 1.3 sessions and above.
 *
 * For further information in MirrorLink 1.2 and below, see section 5.5,
 * 'Key Event Listing Message', in \ref ml_vnc "[ML-VNC]".
 *
 * \see VNCViewerKeyEventListingCallback
 */
typedef struct
{
  /**
   * Array of key sym values describing the valid characters. The length of 
   * this array is given by the eventsCount member.
   */
  vnc_uint32_t const *events;
  /**
   * Number of key events contained in the events member of this struct.
   */
  vnc_uint32_t eventsCount;
} VNCKeyEventListing;

/**
 * \brief Describes the supported key event listing being requested.
 *
 * Key event listing support is deprecated in MirrorLink 1.3.
 * This structure is not used in MirrorLink 1.3 sessions and above.
 *
 * For further information in MirrorLink 1.2 and below, see section 5.5,
 * 'Key Event Listing Message', in \ref ml_vnc "[ML-VNC]".
 *
 * \see VNCKeyEventListingRequest
 */
typedef enum
{
  /** Enable key event listings from the server. */
  VNCKeyEventListingRequestFlagEnabled      = 0x00000001,
  /** Indicate the viewer supports incremental key listing updates. */
  VNCKeyEventListingRequestFlagIncremental  = 0x00000002,
  /** Request that the key event counter is reset. */
  VNCKeyEventListingRequestFlagResetCounter = 0x00000004
} VNCKeyEventListingRequestFlags;

/**
 * \brief Structure holding a KeyEventListingRequest MirrorLink
 * extension message to be sent to the server.
 *
 * To send a KeyEventListingRequest, the viewer application
 * should initialize the structure and then call
 * VNCViewerSendKeyEventListingRequest.
 *
 * Key event listing support is deprecated in MirrorLink 1.3.
 * This structure is not used in MirrorLink 1.3 sessions and above.
 *
 * \see VNCViewerSendKeyEventListingRequest
 */
typedef struct
{
  /**
   * A bit-field representing describing the key event listing request.
   *
   * The value should be a combination of values from the
   * \ref VNCKeyEventListingRequestFlags enumeration.
   */
  VNCKeyEventListingRequestFlags flags;
} VNCKeyEventListingRequest;

/**
 * \brief Used with ::VNCViewerContentAttestationFailureCallback to inform the
 * application of exactly how content attestation has failed.
 *
 * The following errors are notified only if the corresponding bit is set in 
 * ::VNCParameterContentAttestationSignatureFlag:
 *
 *  - ::VNCViewerContentAttestationFailureFramebufferNotAttested
 *  - ::VNCViewerContentAttestationFailureContextInformationNotAttested
 *  - ::VNCViewerContentAttestationFailurePixelCountNotAttested
 *
 * For example, if the application does not request that the framebuffer
 * content hash in included in the content attestation, and the server does not
 * include it, then this is not considered an error. In that case, the
 * VNCViewerContentAttestationFailureCallback would not be not called.
 */
typedef enum
{
  /** The MirrorLink server returned a 'not implemented' error. */
  VNCViewerContentAttestationFailureErrorNotImplemented           = 0x00000001,

  /**
   * The MirrorLink server returned a 'no session key' error.
   *
   * VNC Mobile Viewer SDK always provides a session key in its first
   * ContentAttestationRequest of every session. This error probably indicates
   * a bug in the server implementation.
   */
  VNCViewerContentAttestationFailureErrorNoSessionKey             = 0x00000002,

  /** The MirrorLink server returned an 'other error' error. */
  VNCViewerContentAttestationFailureErrorOther                    = 0x00000004,

  /** The MirrorLink server returned an unrecognized error code. */
  VNCViewerContentAttestationFailureErrorUnknown                  = 0x00000008,

  /** A ContentAttestationResponse was not received in a timely manner. */
  VNCViewerContentAttestationFailureNoResponse                    = 0x00010000,

  /** A ContentAttestationResponse was received, but it was not signed. */
  VNCViewerContentAttestationFailureNoSignature                   = 0x00020000,

  /**
   * A ContentAttestationResponse was received, but it did not contain the hash
   * of the framebuffer content.
   */
  VNCViewerContentAttestationFailureFramebufferNotAttested        = 0x00040000,

  /**
   * A ContentAttestationResponse was received, but it did not contain the hash
   * of the context information.
   */
  VNCViewerContentAttestationFailureContextInformationNotAttested = 0x00080000,

  /**
   * A ContentAttestationResponse was received, but it did not contain the
   * number of updated pixels.
   */
  VNCViewerContentAttestationFailurePixelCountNotAttested         = 0x00100000,
} VNCViewerContentAttestationFailure;

/**
 * \brief Notifies the viewer application that the SDK has received the
 * server's ServerDisplayConfiguration.
 *
 * This callback is optional.  You should implement it if either:
 *
 *  - the viewer application needs to take note of the information in the
 *    ServerDisplayConfiguration message (for example, to check at runtime
 *    which pixel formats the server supports), or
 *  - the viewer application needs to adjust the values in the
 *    ClientDisplayConfiguration message before it is sent to the server (for
 *    example, to request that the server enable server-side framebuffer
 *    scaling or rotation).
 *
 * This callback will only be used when connected to a MirrorLink server via
 * an RFB3 session with MirrorLink support.
 *
 * For more information, see 
 * \ref mirrorlink_handshake "The MirrorLink handshake".
 * 
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pServerDisplayConfiguration A pointer to the decoded
 * VNCServerDisplayConfiguration structure.
 * \param serverDisplayConfigurationSize The size of the structure pointed to
 * by pServerDisplayConfiguration.
 * \param pClientDisplayConfiguration A pointer to the pre-filled
 * VNCClientDisplayConfiguration structure.
 * \param clientDisplayConfigurationSize The size of the structure pointed to
 * by pClientDisplayConfiguration.
 *
 * \see VNCServerDisplayConfiguration, VNCClientDisplayConfiguration
 */
typedef void VNCCALL
VNCViewerServerDisplayConfigurationCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCServerDisplayConfiguration *pServerDisplayConfiguration,
    size_t serverDisplayConfigurationSize,
    VNCClientDisplayConfiguration *pClientDisplayConfiguration,
    size_t clientDisplayConfigurationSize);

/**
 * \brief Notifies the viewer application that the SDK has received the server's
 * ServerEventConfiguration.
 *
 * This callback is optional.  You should implement it if either:
 *
 *  - the viewer application needs to take note of the information in the
 *    ServerEventConfiguration message (for example, to check at runtime
 *    which button inputs the server supports), or
 *  - the viewer application needs to adjust the values in the
 *    VNCClientEventConfiguration structure before it is sent to the server
 *    (for example, to request support for particular button inputs), or
 *  - the viewer application needs to adjust the values in the
 *    VNCDeviceStatusRequest structure before it is sent to the server (for
 *    example, to request that The server rotates the framebuffer).
 *
 * This callback will only be used when connected to a MirrorLink server via
 * an RFB3 session with MirrorLink support.
 *
 * For more information, see 
 * \ref mirrorlink_handshake "The MirrorLink handshake".
 * See also 
 * \ref mirrorlink_status "DeviceStatus and DeviceStatusRequest messages"
 * for an explanation of why this callback provides a VNCDeviceStatusRequest
 * structure pointer.
 *
 * The Viewer SDK will base the pre-filled event configuration on whether the
 * Server supports knob events, or not.  If knob events are supported, then the
 * SDK will map, in VNCViewerSendDeviceKeyEvent, the OK and left, right, up,
 * down device keys to the knob events and it will not advertise support for the
 * OK device key the pre-filled configuration. Otherwise the device keys will be
 * advertised. A Server is considered to support knob events if it supports at
 * least shift on X axis and Y axis and push on Z axis.
 *
 * The SDK will pre-fill the event configuration with support for all device
 * keys that the server supports, except the OK key if the knob is supported.
 *
 * If the viewer application supports MirrorLink 1.1 or above, and does not
 * support pointer events, then it is required to support certain knob key
 * events. From MirrorLink 1.3 onwards, viewer applications not supporting
 * pointer events must additionally support the back device key. See
 * \ref VNCClientEventConfiguration for further information.
 *
 * Some miscellaneous key support features have been deprecated in MirrorLink
 * 1.3. The viewer application should not attempt to use these features if it
 * supports MirrorLink 1.3 or above. See \ref VNCMiscKeySupport for further
 * information.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pServerEventConfiguration A pointer to the decoded
 * VNCServerEventConfiguration structure.
 * \param serverEventConfigurationSize The size of the structure pointed to
 * by pServerEventConfiguration.
 * \param pClientEventConfiguration A pointer to the pre-filled
 * VNCClientEventConfiguration structure.
 * \param clientEventConfigurationSize The size of the structure pointed to
 * by pClientEventConfiguration.
 * \param pFirstDeviceStatusRequest A pointer to the pre-filled
 * VNCDeviceStatusRequest structure.
 * \param firstDeviceStatusRequestSize The size of the structure pointed to by
 * pFirstDeviceStatusRequest.
 *
 * \see VNCServerEventConfiguration, VNCClientEventConfiguration,
 * VNCDeviceStatusRequest, VNCDeviceStatus
 */
typedef void VNCCALL
VNCViewerServerEventConfigurationCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCServerEventConfiguration *pServerEventConfiguration,
    size_t serverEventConfigurationSize,
    VNCClientEventConfiguration *pClientEventConfiguration,
    size_t clientEventConfigurationSize,
    VNCDeviceStatusRequest *pFirstDeviceStatusRequest,
    size_t firstDeviceStatusRequestSize);

/**
 * \brief Notifies the viewer application that the SDK has received a
 * DeviceStatus MirrorLink extension message from the server.
 * 
 * This callback is optional.  You should implement it if the viewer
 * application uses VNCViewerSendDeviceStatusRequest() to send
 * DeviceStatusRequest messages to the server, and you need to be informed of
 * the results.
 *
 * This callback will only be used when connected to a MirrorLink server via
 * an RFB3 session with MirrorLink support.
 *
 * For more information, see
 * \ref mirrorlink_status "DeviceStatus and DeviceStatusRequest messages".
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pDeviceStatus A pointer to the decoded VNCDeviceStatus structure that
 * has been received from the server.
 * \param deviceStatusRequestSize The size of the structure pointed to by
 * pDeviceStatusRequest.
 *
 * \see VNCViewerSendDeviceStatusRequest, VNCDeviceStatusRequest,
 * VNCDeviceStatus
 */
typedef void VNCCALL
VNCViewerDeviceStatusCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCDeviceStatus *pDeviceStatus,
    size_t deviceStatusSize);

/**
 * \brief Notifies the viewer application that the SDK has received a
 * MirrorLink Context Information rectangle from the server.
 * 
 * This callback is optional.  You should implement it if the viewer
 * application needs to be kept informed of which applications are being
 * displayed by the server (for example, to implement a driver distraction
 * policy).
 *
 * This callback will only be used when connected to a MirrorLink server via
 * an RFB3 session with MirrorLink support.
 *
 * For more information, see
 * \ref mirrorlink_context "Context Information and driver distraction".
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pRectangle The rectangle within the server display to which this
 * notification applies.  This may be the entire server display, or, in the
 * case of screens composed of multiple applets (e.g. home screens), a smaller
 * area within it.
 * \param pContextInformation A pointer to the decoded VNCContextInformation
 * structure that has been received from the server.
 * \param contextInformationSize The size of the structure pointed to by
 * pContextInformation.
 *
 * \see VNCRectangle, VNCContextInformation
 */
typedef void VNCCALL
VNCViewerContextInformationCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCRectangle *pRectangle,
    const VNCContextInformation *pContextInformation,
    size_t contextInformationSize);

/**
 * \brief Notifies the application that the SDK has received a MirrorLink High
 * Speed Media Link (HSML) Information rectangle from the server.
 *
 * This callback is optional. You should implement it if the viewer application
 * supports HSML. When this callback is first invoked, the viewer application
 * must establish an HSML connection with the MirrorLink server. The VNC HSML
 * SDK may be used to accomplish this.
 *
 * This callback will only be used when VNCParameterHSML has been set to enable
 * HSML support, and the viewer is connected to a MirrorLink server, via an
 * RFB3 session with MirrorLink support, that also supports HSML.
 *
 * For more information, see \ref mirrorlink_hsml.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pRectangle The rectangle representing the current dimensions of the
 * server's display. The SDK shall call VNCViewerDesktopResizeCallback() when
 * the server's display dimensions change.
 * \param pIdentifier The UUID of the MirrorLink server's HSML source. The
 * viewer application should match this with the identifier reported in the
 * corresponding HSML connection.
 * \param identifierSize The size of pIdentifier.
 *
 * \see VNCParameterHSML
 */
typedef void VNCCALL
VNCViewerHSMLInformationCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCRectangle *pRectangle,
    const vnc_uint8_t *pIdentifier,
    size_t identifierSize);

/**
 * \brief Notifies the application that the SDK has received a
 * VirtualKeyboardTrigger MirrorLink extension message from the server.
 *
 * This callback is optional. It is only needed if a request to
 * enable virtual keyboard support has been sent to the server by calling
 * \ref VNCViewerSendVirtualKeyboardTriggerRequest(). This callback is not
 * used in MirrorLink 1.3 sessions and above.
 *
 * For more information, see
 * \ref mirrorlink_virtual_kbd "Virtual Keyboard Triggers".
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pTriggerInformation The virtual keyboard information from the server.
 * \param triggerInformationSize The size of the structure pointed to by pTriggerInformation.
 *
 * \see VNCVirtualKeyboardTrigger, VNCViewerSendVirtualKeyboardTriggerRequest
 */
typedef void VNCCALL
VNCViewerVirtualKeyboardTriggerCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCVirtualKeyboardTrigger *pTriggerInformation,
    size_t triggerInformationSize);

/**
 * \brief Notifies the application that the SDK has received an
 * EventMapping MirrorLink extension message from the server.
 *
 * This callback is optional. It is not used in MirrorLink 1.3 sessions and
 * above.
 *
 * For more information, see
 * \ref mirrorlink_event_mapping "Event Mapping"
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pEventMappingInformation The event mapping information from the server.
 * \param eventMappingInformationSize The size of the structure pointed to by pEventMappingInformation.
 *
 * \see VNCEventMapping, VNCViewerSendEventMappingRequest
 */
typedef void VNCCALL
VNCViewerEventMappingCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCEventMapping *pEventMappingInformation,
    size_t eventMappingInformationSize);

/**
 * \brief Notifies the application that the SDK has received a
 * change to the list of valid key symbols from the server.
 *
 * This callback is optional. It is not used in MirrorLink 1.3 sessions and
 * above.
 *
 * For more information, see
 * \ref mirrorlink_key_event_listing "Key Event Listing".
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param pKeyEventListing The key event listing information from the server.
 * \param keyEventListingSize The size of the structure pointed to by pKeyEventListing.
 *
 * \see VNCKeyEventListing, VNCViewerSendKeyEventListingRequest
 */
typedef void VNCCALL
VNCViewerKeyEventListingCallback(
    VNCViewer *pViewer,
    void *pContext,
    const VNCKeyEventListing *pKeyEventListing,
    size_t keyEventListingSize);

/**
 * \brief Notifies the application that MirrorLink content attestation has
 * failed, or that the server has not attested everything that was requested.
 *
 * This callback is called by the SDK if content attestation fails in a way
 * that the client application may consider to be recoverable. See \ref
 * mirrorlink_ca_callback for details of how the SDK uses this callback.
 *
 * The callback should inspect the failure parameter and then call
 * VNCViewerContentAttestationFailureResult() to inform the SDK of the outcome.
 * This call may be made either from within the callback or at a later time or
 * on another thread. The session will not continue until
 * VNCViewerContentAttestationFailureResult() has been called.
 *
 * This callback is optional. If no VNCViewerContentAttestationFailureCallback
 * is provided, then the SDK considers all content attestation failures to be
 * fatal to the current session.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param failure A combination of values from the
 * ::VNCViewerContentAttestationFailure 
 */
typedef void VNCCALL
VNCViewerContentAttestationFailureCallback(
        VNCViewer *pViewer,
        void *pContext,
        VNCViewerContentAttestationFailure failure);

/**
 * \brief Requests a locale dependent conversion of a character to upper case.
 *
 * This callback is called by the SDK when a locale dependent case conversion
 * is needed.
 *
 * It is the responsibility of the viewer to pick a suitable locale for the
 * conversion. The locale of the server is provided in the
 * VNCViewerServerEventConfigurationCallback. It is recommended to pick a
 * locale as close as possible to the server locale when performing these
 * conversions.
 *
 * This callback is optional. If no VNCViewerToUpperCallback is provided then
 * the SDK will use the default system locale to perform the case conversion.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param character The unicode value of the character to convert.
 * \return The upper case form of character, or character if no case
 * conversion is necessary
 */
typedef vnc_uint32_t VNCCALL
VNCViewerToUpperCallback(
        VNCViewer *pViewer,
        void *pContext,
        vnc_uint32_t character);

/**
 * \brief Requests a locale dependent conversion of a character to lower case.
 *
 * This callback is called by the SDK when a locale dependent case conversion
 * is needed.
 *
 * It is the responsibility of the viewer to pick a suitable locale for the
 * conversion. The locale of the server is provided in the
 * VNCViewerServerEventConfigurationCallback. It is recommended to pick a
 * locale as close as possible to the server locale when performing these
 * conversions.
 *
 * This callback is optional. If no VNCViewerToLowerCallback is provided then
 * the SDK will use the default system locale to perform the case conversion.
 *
 * \param pViewer The VNC Viewer instance to which this notification applies.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param character The unicode value of the character to convert.
 * \return The lower case form of character, or character if no case
 * conversion is necessary
 */
typedef vnc_uint32_t VNCCALL
VNCViewerToLowerCallback(
        VNCViewer *pViewer,
        void *pContext,
        vnc_uint32_t character);

/**
 * \brief Reports the outcome of calling
 * VNCViewerSendFramebufferBlockingNotificationEx.
 *
 * The Viewer SDK will report the outcome of the
 * FramebufferBlockingNotification request through this callback.
 *
 * \param pViewer The VNC viewer used for sending the
 * FramebufferBlockingNotification.
 * \param pContext The context pointer associated with the VNC Viewer by
 * VNCViewerSetContext().
 * \param outcome Indicates whether the FramebufferBlockingNotification was
 * sent to the server or not. The possible values are:
 *   - VNCViewerErrorNone The message was sent to the server successfully.
 *   Please note that this is a guarantee that the message has been buffered to
 *   be sent to the server, it is <i>not</i> a guarantee that the server has
 *   received it.
 *   - VNCViewerErrorReset The message was not sent because the client has
 *   cancelled sending (through
 *   VNCViewerCancelFramebufferBlockingNotifications), a desktop resize was
 *   received (which forces the cancellation), or new context information has
 *   been received.
 *   - VNCViewerErrorDisconnected The message was not sent because the
 *   connection to the server has stopped.
 *   - VNCViewerErrorIllegalWhileNotRunning The viewer thread is not running.
 *   - VNCViewerErrorIllegalBeforeServerInit A message of this type may not be
 *   sent until the MirrorLink handshake is complete.
 *   - VNCViewerErrorNoProtocolSupport The message cannot be sent because the
 *   server is not a MirrorLink server.
 *   - VNCViewerErrorNotImplemented This build of the SDK does not contain
 *   support for MirrorLink.
 *   - VNCViewerErrorDeprecatedFieldUsed One or more framebuffer blocking
 *   reasons are invalid, as they have been deprecated in the MirrorLink
 *   version negotiated with the VNC Server.
 * \param pRectangle The rectangle to which the FramebufferBlockingNotification
 * applies.
 * \param pFramebufferBlockingNotification Structure holding the
 * FramebufferBlockingNotification that was sent to the server.
 * \param framebufferBlockingNotificationSize The size of the structure pointed
 * to by pFramebufferBlockingNotification.
 *
 * \see VNCViewerSendFramebufferBlockingNotificationEx,
 * VNCViewerCancelFramebufferBlockingNotifications
 */
typedef void VNCCALL
VNCViewerFramebufferBlockingNotificationSentCallback(
    VNCViewer *pViewer,
    void *pContext,
    VNCViewerError outcome,
    const VNCRectangle *pRectangle,
    const VNCFramebufferBlockingNotification *pFramebufferBlockingNotification,
    size_t framebufferBlockingNotificationSize);

/**
 * \brief Sends a DeviceStatusRequest MirrorLink extension message to the
 * server.
 *
 * If the viewer application needs to be notified of the server's response, it
 * should implement a VNCViewerDeviceStatusCallback.
 *
 * For more information, see
 * \ref mirrorlink_status "DeviceStatus and DeviceStatusRequest messages".
 *
 * Some device features have been deprecated in MirrorLink 1.3. The viewer
 * application should not attempt to use deprecated feature status values if it
 * supports MirrorLink 1.3 or above. See \ref VNCDeviceStatusFeature for
 * further information.
 *
 * \param pViewer The VNC viewer that should send the DeviceStatusRequest.
 * \param pDeviceStatusRequest Structure holding the DeviceStatusRequest to be
 * sent to the server.
 * \param deviceStatusRequestSize The size of the structure pointed to by 
 * pDeviceStatusRequest.
 *
 * \retval VNCViewerErrorNone The message was sent to the server successfully.
 * \retval VNCViewerErrorIllegalWhileNotRunning The viewer thread is not
 * running.
 * \retval VNCViewerErrorIllegalBeforeServerInit A message of this type may not
 * be sent until the MirrorLink handshake is complete.
 * \retval VNCViewerErrorNoProtocolSupport The message cannot be sent because
 * the server is not a MirrorLink server.
 * \retval VNCViewerErrorNotImplemented This build of the SDK does not contain
 * support for MirrorLink.
 * \retval VNCViewerErrorDeprecatedFieldUsed The requested status of one or
 * more features is invalid, as those features have been deprecated in the
 * MirrorLink version negotiated with the VNC Server.
 *
 * \see VNCDeviceStatusRequest, VNCViewerDeviceStatusCallback,
 * VNCDeviceStatus
 *
 * \section mirrorlink_legal Legal information
 *
 * MirrorLink is a registered trademark of the Car Connectivity Consortium.
 */
typedef VNCViewerError VNCCALL
VNCViewerSendDeviceStatusRequest(
    VNCViewer *pViewer,
    VNCDeviceStatusRequest *pDeviceStatusRequest,
    size_t deviceStatusRequestSize);

/**
 * \brief Sends a FramebufferBlockingNotification MirrorLink extension
 * message to the server.
 *
 * For more information, see
 * \ref mirrorlink_blocking "FramebufferBlockingNotification and AudioBlockingNotification messages".
 *
 * As it is difficult to get the timing right to match the requirements of the
 * MirrorLink specification, it is recommended to use
 * VNCViewerSendFramebufferBlockingNotificationEx instead. This would ensure
 * that the ViewerSDK will send the blocking notification at the correct step.
 *
 * Some framebuffer blocking reasons have been deprecated in MirrorLink 1.3.
 * The viewer application should not specify these reasons if it supports
 * MirrorLink 1.3 or above. See \ref VNCFramebufferBlockReason for further
 * information.
 *
 * \param pViewer The VNC viewer that should send the
 * FramebufferBlockingNotification.
 * \param pRectangle The rectangle to which the FramebufferBlockingNotification
 * applies.  If this FramebufferBlockingNotification is being sent as a result
 * of a VNCViewerContextInformationCallback, then *pRectangle must be equal to
 * the rectangle passed to that callback.
 * \param pFramebufferBlockingNotification Structure holding the
 * FramebufferBlockingNotification to be send to the server.
 * \param framebufferBlockingNotificationSize The size of the structure pointed
 * to by pFramebufferBlockingNotification.
 *
 * \retval VNCViewerErrorNone The message was sent to the server successfully.
 * \retval VNCViewerErrorIllegalWhileNotRunning The viewer thread is not
 * running.
 * \retval VNCViewerErrorIllegalBeforeServerInit A message of this type may not
 * be sent until the MirrorLink handshake is complete.
 * \retval VNCViewerErrorNoProtocolSupport The message cannot be sent because
 * the server is not a MirrorLink server.
 * \retval VNCViewerErrorNotImplemented This build of the SDK does not contain
 * support for MirrorLink.
 * \retval VNCViewerErrorDeprecatedFieldUsed One or more framebuffer blocking
 * reasons are invalid, as they have been deprecated in the MirrorLink version
 * negotiated with the VNC Server.
 *
 * \see VNCViewerSendFramebufferBlockingNotificationEx, VNCRectangle,
 * VNCFramebufferBlockingNotification
 */
typedef VNCViewerError VNCCALL
VNCViewerSendFramebufferBlockingNotification(
    VNCViewer *pViewer,
    VNCRectangle *pRectangle,
    VNCFramebufferBlockingNotification *pFramebufferBlockingNotification,
    size_t framebufferBlockingNotificationSize);

/**
 * \brief Sends an AudioBlockingNotification MirrorLink extension message to
 * the server.
 *
 * For more information, see
 * \ref mirrorlink_blocking "FramebufferBlockingNotification and AudioBlockingNotification messages".

 * Some audio blocking reasons have been deprecated in MirrorLink 1.3. The
 * viewer application should not specify these reasons if it supports
 * MirrorLink 1.3 or above. See \ref VNCAudioBlockReason for further
 * information.
 *
 * \param pViewer The VNC viewer that should send the
 * AudioBlockingNotification.
 * \param pAudioBlockingNotification Structure holding the
 * AudioBlockingNotification to be send to the server.
 * \param audioBlockingNotificationSize The size of the structure pointed to by
 * pAudioBlockingNotification.
 *
 * \retval VNCViewerErrorNone The message was sent to the server successfully.
 * \retval VNCViewerErrorIllegalWhileNotRunning The viewer thread is not
 * running.
 * \retval VNCViewerErrorIllegalBeforeServerInit A message of this type may not
 * be sent until the MirrorLink handshake is complete.
 * \retval VNCViewerErrorNoProtocolSupport The message cannot be sent because
 * the server is not a MirrorLink server.
 * \retval VNCViewerErrorNotImplemented This build of the SDK does not contain
 * support for MirrorLink.
 * \retval VNCViewerErrorDeprecatedFieldUsed One or more audio blocking reasons
 * are invalid, as they have been deprecated in the MirrorLink version
 * negotiated with the VNC Server.
 *
 * \see VNCAudioBlockingNotification
 */
typedef VNCViewerError VNCCALL
VNCViewerSendAudioBlockingNotification(
    VNCViewer *pViewer,
    VNCAudioBlockingNotification *pAudioBlockingNotification,
    size_t audioBlockingNotificationSize);

/**
 * \brief Sends a VirtualKeyboardTriggerRequest MirrorLink extension message to
 * the server.
 *
 * For more information, see
 * \ref mirrorlink_virtual_kbd "Virtual Keyboard Triggers".
 *
 * This message is not supported in MirrorLink 1.3 sessions and above.
 *
 * \param pViewer The VNC viewer that should send the
 * VirtualKeyboardTriggerRequest.
 * \param pVirtualKeyboardTriggerRequest Structure holding the
 * VirtualKeyboardTriggerRequest to be sent to the server.
 * \param virtualKeyboardTriggerRequestSize The size of the structure
 * pointed to by pVirtualKeyboardTriggerRequest.
 *
 * \retval VNCViewerErrorNone The message was sent to the server successfully.
 * \retval VNCViewerErrorIllegalWhileNotRunning The viewer thread is not
 * running.
 * \retval VNCViewerErrorIllegalBeforeServerInit A message of this type may not
 * be sent until the MirrorLink handshake is complete.
 * \retval VNCViewerErrorNoProtocolSupport The message cannot be sent because
 * the server is not a MirrorLink server, or the version of MirrorLink
 * negotiated for the session is 1.3 or above.
 * \retval VNCViewerErrorNotImplemented This build of the SDK does not contain
 * support for MirrorLink.
 *
 * \see VNCVirtualKeyboardTriggerRequest, VNCViewerVirtualKeyboardTriggerCallback
 */
typedef VNCViewerError VNCCALL
VNCViewerSendVirtualKeyboardTriggerRequest(
    VNCViewer *pViewer,
    VNCVirtualKeyboardTriggerRequest *pVirtualKeyboardTriggerRequest,
    size_t virtualKeyboardTriggerRequestSize);

/**
 * \brief Sends an EventMappingRequest MirrorLink extension message to
 * the server.
 *
 * For more information, see
 * \ref mirrorlink_event_mapping "Event Mapping".
 *
 * This message is not supported in MirrorLink 1.3 sessions and above.
 *
 * \param pViewer The VNC viewer that should send the
 * VirtualKeyboardTriggerRequest.
 * \param pEventMappingRequest Structure holding the
 * EventMappingRequest to be sent to the server.
 * \param eventMappingRequestSize The size of the structure
 * pointed to by pEventMappingRequest.
 *
 * \retval VNCViewerErrorNone The message was sent to the server successfully.
 * \retval VNCViewerErrorIllegalWhileNotRunning The viewer thread is not
 * running.
 * \retval VNCViewerErrorIllegalBeforeServerInit A message of this type may not
 * be sent until the MirrorLink handshake is complete.
 * \retval VNCViewerErrorNoProtocolSupport The message cannot be sent because
 * the server is not a MirrorLink server, or the version of MirrorLink
 * negotiated for the session is 1.3 or above.
 * \retval VNCViewerErrorNotImplemented This build of the SDK does not contain
 * support for MirrorLink.
 *
 * \see VNCEventMappingRequest, VNCViewerEventMappingCallback
 */
typedef VNCViewerError VNCCALL
VNCViewerSendEventMappingRequest(
    VNCViewer *pViewer,
    VNCEventMappingRequest *pEventMappingRequest,
    size_t eventMappingRequestSize);

/**
 * \brief Sends a KeyEventListingRequest MirrorLink extension message to
 * the server.
 *
 * For more information, see
 * \ref mirrorlink_key_event_listing "Key Event Listing".
 *
 * This message is not supported in MirrorLink 1.3 sessions and above.
 *
 * \param pViewer The VNC viewer that should send the
 * KeyEventListingRequest.
 * \param pKeyEventListingRequest Structure holding the
 * KeyEventListingRequest to be sent to the server.
 * \param keyEventListingRequestSize The size of the structure
 * pointed to by pKeyEventListingRequest.
 *
 * \retval VNCViewerErrorNone The message was sent to the server successfully.
 * \retval VNCViewerErrorIllegalWhileNotRunning The viewer thread is not
 * running.
 * \retval VNCViewerErrorIllegalBeforeServerInit A message of this type may not
 * be sent until the MirrorLink handshake is complete.
 * \retval VNCViewerErrorNoProtocolSupport The message cannot be sent because
 * the server is not a MirrorLink server, or the version of MirrorLink
 * negotiated for the session is 1.3 or above.
 * \retval VNCViewerErrorNotImplemented This build of the SDK does not contain
 * support for MirrorLink.
 *
 * \see VNCKeyEventListingRequest, VNCViewerKeyEventListingCallback
 */
typedef VNCViewerError VNCCALL
VNCViewerSendKeyEventListingRequest(
    VNCViewer *pViewer,
    VNCKeyEventListingRequest *pKeyEventListingRequest,
    size_t keyEventListingRequestSize);

/**
 * Informs the SDK of the outcome of a
 * ::VNCViewerContentAttestationFailureCallback.
 *
 * If result != 0, then the session continues. The
 * ::VNCViewerContentAttestationFailureCallback will not be called again
 * within the same session until a failure of a different kind occurs.
 *
 * For example it is possible for
 * ::VNCViewerContentAttestationFailureCallback to be called with a
 * failure reason of
 * ::VNCViewerContentAttestationFailureContextInformationNotAttested
 * followed by a call with a failure reason of
 * ::VNCViewerContentAttestationFailurePixelCountNotAttested. It is
 * then possible for the callback to be called again should a failure
 * of
 * ::VNCViewerContentAttestationFailureContextInformationNotAttested
 * occur after the
 * ::VNCViewerContentAttestationFailureContextInformationNotAttested
 * failure.
 *
 * If result == 0, then session will terminate with error
 * ::VNCViewerErrorContentAttestationFailed.
 *
 * \param pViewer The VNC viewer that called the
 * VNCViewerContentAttestationFailureCallback.
 * \param result If zero, the session should be terminated. 
 */
typedef void VNCCALL
VNCViewerContentAttestationFailureResult(VNCViewer *pViewer,
                                         vnc_int32_t result);

/**
 * \brief Sends a FramebufferBlockingNotification MirrorLink extension
 * message to the server as soon as the client is allowed to send.
 *
 * When the MirrorLink client decides that the framebuffer is not appropiate to
 * be presented to the user, for example if driving mode is enabled, it should
 * inform the server that it is blocking the remote display. The timing for
 * sending the blocking notification is dependent on when the framebuffer
 * update is received from the server. The advantage of using this method,
 * instead of VNCViewerSendFramebufferBlockingNotification, is that the Viewer
 * SDK will send the notification at the right time, rather than straight away.
 *
 * For more information, see
 * \ref mirrorlink_blocking "FramebufferBlockingNotification and AudioBlockingNotification messages".
 *
 * The application will be notified when the notification has been sent to the
 * server in the VNCViewerFramebufferBlockingNotificationSentCallback.
 *
 * If the client decides to unblock the framebuffer, it should call
 * VNCViewerCancelFramebufferBlockingNotifications.
 *
 * If a desktop resize is received before the blocking notification can be
 * sent, then the SDK will automatically cancel the notification. The client
 * may re-send a notification once it receives the new context information for
 * the new size.
 *
 * If a blocking notification has been queued on previously received context
 * information and new context information is received before the SDK has the
 * chance to send the blocking notification, then the SDK will automatically
 * cancel the old blocking notifications (and report error
 * VNCViewerErrorReset). Therefore the client application must ensure it sends
 * blocking notifications when it receives any context information that it
 * plans to block.
 *
 * Some framebuffer blocking reasons have been deprecated in MirrorLink 1.3.
 * The viewer application should not specify these reasons if it supports
 * MirrorLink 1.3 or above. See \ref VNCFramebufferBlockReason for further
 * information.
 *
 * \param pViewer The VNC viewer that should send the
 * FramebufferBlockingNotification.
 * \param pRectangle The rectangle to which the FramebufferBlockingNotification
 * applies.  If this FramebufferBlockingNotification is being sent as a result
 * of a VNCViewerContextInformationCallback, then *pRectangle must be equal to
 * the rectangle passed to that callback.
 * \param pFramebufferBlockingNotification Structure holding the
 * FramebufferBlockingNotification to be send to the server.
 * \param framebufferBlockingNotificationSize The size of the structure pointed
 * to by pFramebufferBlockingNotification.
 *
 * \see VNCViewerFramebufferBlockingNotificationSentCallback,
 * VNCViewerCancelFramebufferBlockingNotifications, VNCRectangle,
 * VNCFramebufferBlockingNotification
 */
typedef void VNCCALL
VNCViewerSendFramebufferBlockingNotificationEx(
    VNCViewer *pViewer,
    const VNCRectangle *pRectangle,
    const VNCFramebufferBlockingNotification *pFramebufferBlockingNotification,
    size_t framebufferBlockingNotificationSize);

/**
 * \brief Cancels all the framebuffer blocking notifications that are queued to
 * be sent to the server.
 *
 * This cancels all the blocking notifications made through
 * VNCViewerSendFramebufferBlockingNotificationEx. The
 * VNCViewerFramebufferBlockingNotificationSentCallback will report
 * VNCViewerErrorReset.
 *
 * \param pViewer The VNC viewer used for sending the
 * FramebufferBlockingNotification.
 *
 * \see VNCViewerFramebufferBlockingNotificationSentCallback,
 * VNCViewerSendFramebufferBlockingNotificationEx
 */
typedef void VNCCALL
VNCViewerCancelFramebufferBlockingNotifications(VNCViewer *pViewer);

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCMIRRORLINK_H__) */
